<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreServiceGroupRequest;
use App\Models\Service;
use App\Models\ServiceGroup;
use App\Traits\HttpResponses;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
/**
 * @group Service group management
 *
 * APIs to manage service group resources
 * */
class ServiceGroupController extends Controller
{
    use HttpResponses;
    /**
     * Display a listing of the resource.
     */
    public function index(): JsonResponse
    {
        $serviceGroup = ServiceGroup::all();
        return response()->json($serviceGroup);
    }
    /**
     * Display the specified resource.
     */
    public function show($id): JsonResponse
    {
        try {
            $serviceGroup = ServiceGroup::findOrFail($id);
            return response()->json($serviceGroup);

        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no service group with this ID", 404);
        }
    }

    /**
     * @apiResourceCollection App\Http\Resources\ServiceGroupCollection
     * @apiResourceModel App\Models\ServiceGroup
     * @response 200 scenario="the service group created successfully"
     * {
     * "status": "Request was successful",
     * "message": "the service group has been created successfully",
     * "data": {
     * "code": "45123",
     * "name": "surgery",
     * "updated_at": "2024-10-11T07:35:31.000000Z",
     * "created_at": "2024-10-11T07:35:31.000000Z",
     * "id": 2
     * }
     * }
     * @response 422 scenario="required parameters missing"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "code": [
     * "the service group code is required"
     * ],
     * "name": [
     * "service group name is required"
     * ]
     * }
     * }
     * @response 404 scenario="code or name already exist"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "code": [
     * "the specified code already exists"
     * ],
     * "name": [
     * "the specified name already exists"
     * ]
     * }
     * }
     *
     */
    public function store(StoreServiceGroupRequest $request):JsonResponse
    {
        $validatedData=$request->validated();
        $serviceGroup=ServiceGroup::create($validatedData);
        return $this->success($serviceGroup,"the service group has been created successfully");
    }

    /**
     * @param StoreServiceGroupRequest $request
     * @param $id
     * @return JsonResponse
     * @response 200 scenario="the service group has been update successfully"
     * {
     * "status": "Request was successful",
     * "message": "the service group has been updated successfully",
     * "data": {
     * "code": "54564512",
     * "name": "surgery"
     * }
     * }
     * @response 404 scenario="code or name already exist"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "code": [
     * "the specified code already exists"
     * ],
     * "name": [
     * "the specified name already exists"
     * ]
     * }
     * }
     * @response 404 scenario="There is no service group with the specified ID"
     * {
     * "status": "error",
     * "message": "There is no service group with this ID",
     * "data": ""
     * }
     */
    public function update(StoreServiceGroupRequest $request, $id): JsonResponse
    {
        try {
            $serviceGroup = ServiceGroup::findOrFail($id);
            $validatedData = $request->validated();
            $serviceGroup->update($validatedData);
            return $this->success($validatedData, "the service group has been updated successfully");
        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no service group with this ID", 404);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @response 404 scenario="There is no service group with the specified ID"
     *  {
     *  "status": "error",
     *  "message": "There is no service group with this ID",
     *  "data": ""
     *  }
     * @response 200 scenario="the service group has been deleted successfully"
     * {
     * "status": "Request was successful",
     * "message": "the service group has been  deleted successfully",
     * "data": null
     * }
     */
    public function destroy($id): JsonResponse
    {
        try {
            $serviceGroup = ServiceGroup::findOrFail($id);
            $serviceGroup->delete();
            return $this->success(null,"the service group has been  deleted successfully");

        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no service group with this ID", 404);
        }
    }
}
