<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreServiceRequest;
use App\Models\Service;
use App\Traits\HttpResponses;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
/**
 * @group Services management
 *
 * APIs to manage services resources
 * */
class ServiceController extends Controller
{
    use HttpResponses;
/**
 * get all services
 * @response 200 "request was successful"{
 * "status": "Request was successful",
 * "message": "success",
 * "data": {
 * "name_english": "teeth removal",
 * "name_arabic": "خلع أسنان",
 * "code": "05304545dssd555223",
 * "QTY": "20",
 * "cost": "210",
 * "price": "260",
 * "service_group_id": "2",
 * "updated_at": "2024-10-11T07:53:18.000000Z",
 * "created_at": "2024-10-11T07:53:18.000000Z",
 * "id": 4
 * }
 * }
 *
 */
    public function index(): JsonResponse
    {
        $service = Service::all();
//        return response()->json($service);
        return $this->success($service,"success");

    }

    /**
     * Store services
     *
     * @response 200 "service group created successfully"{
     * "status": "Request was successful",
     * "message": "the service group has been created successfully",
     * "data": {
     * "code": "4512",
     * "name": "teeth",
     * "updated_at": "2024-10-11T07:48:23.000000Z",
     * "created_at": "2024-10-11T07:48:23.000000Z",
     * "id": 5
     * }
     * }
     * @response 404 scenario="no service group with the specified id"{
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "service_group_id": [
     * "there is no service group with this id"
     * ]
     * }
     * }
     * @response 4003 "code or name already exists"{
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "code": [
     * "the specified code already exists"
     * ],
     * "name": [
     * "the specified name already exists"
     * ]
     * }
     * }
     */
    public function store(StoreServiceRequest $request):JsonResponse
    {
        $validatedData=$request->validated();
//        dd($request->all());
        $service = Service::create($validatedData);
        return $this->success($service,'success');

    }


    /**
     * Display Service by id
     * @response 404 scenario="no service with the specified id"
     * {
     * "status": "error",
     * "message": "There is no service with this ID",
     * "data": ""
     * }
     * @response 200 scenario="request was successful"
     * {
     * "status": "Request was successful",
     * "message": "success",
     * "data": {
     * "id": 4,
     * "service_group_id": 2,
     * "code": "05304545d555223",
     * "name_english": "teeth removal",
     * "name_arabic": "خلع أسنان",
     * "QTY": 20,
     * "cost": 210,
     * "price": 260,
     * "created_at": "2024-10-11T07:53:18.000000Z",
     * "updated_at": "2024-10-11T07:53:18.000000Z"
     * }
     * }
     */
    public function show($id): JsonResponse
    {
        try {
            $service = Service::findOrFail($id);
//            return response()->json($service);
            return $this->success($service,"success");

        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no service with this ID", 404);
        }
    }

    /**
     * update services
     *
     * @response 200 "service group created successfully"{
     * "status": "Request was successful",
     * "message": "the service group has been created successfully",
     * "data": {
     * "code": "4512",
     * "name": "teeth",
     * "updated_at": "2024-10-11T07:48:23.000000Z",
     * "created_at": "2024-10-11T07:48:23.000000Z",
     * "id": 5
     * }
     * }
     * @response 404 scenario="no service group with the specified id"{
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "service_group_id": [
     * "there is no service group with this id"
     * ]
     * }
     * }
     * @response 4003 "code or name already exists"{
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "code": [
     * "the specified code already exists"
     * ],
     * "name": [
     * "the specified name already exists"
     * ]
     * }
     * }
     * @response 404 scenario="there is no service with the specified id"{
     * "status": "error",
     * "message": "There is no service with this ID",
     * "data": ""
     * }
     */
    public function update(StoreServiceRequest $request, $id): JsonResponse
    {
        try {
            // Attempt to find the service by ID
            $service = Service::findOrFail($id);

            // If found, validate and update the service
            $validatedData = $request->validated();
            $service->update($validatedData);

            // Return success response
            return $this->success($validatedData, "Service updated successfully");

        } catch (ModelNotFoundException $e) {
            // If no service is found, return a 404 error response
            return $this->error("", "There is no service with this ID", 404);
        }
    }
    /**
     * delete service
     * @response 200 scenario="request was successful"
     * {
     * "status": "Request was successful",
     * "message": "the service has been deleted successfully",
     * "data": null
     * }
     * @response 404 scenario="there is no service with the specified id"
     * {
     * "status": "error",
     * "message": "There is no service with this ID",
     * "data": ""
     * }
     */
    public function destroy($id): JsonResponse
    {
        try {
            $service = Service::findOrFail($id);
            $service->delete();
            return $this->success(null,"the service has been deleted successfully");

        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no service with this ID", 404);
        }
    }

}
