<?php

namespace App\Http\Controllers;
// use http\Client;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Hesabe\Payment\HesabeCrypt;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
/**
 * @group Hesabe payment gateway management
 *
 * APIs to manage Hesabe payment gateway resources
 * */
class PaymentController extends Controller
{
    protected $hesabeCrypt;

    public function __construct()
    {
        $this->hesabeCrypt = new HesabeCrypt();
    }

    public function createPayment(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|between:0.200,100000|regex:/^\d+(\.\d{1,3})?$/',
        ]);

        $requestData = [
            'merchantCode' => env('HESABE_MERCHANT_CODE'),
            'amount' => $request->amount,
            'currency' => 'KWD',
            'paymentType' => '1', // 1 for KNET
            'responseUrl' => route('payment.response'),
            'failureUrl' => route('payment.failure'),
            'version' => '2.0',
            'orderReferenceNumber' => uniqid(),
        ];

        $requestDataJson = json_encode($requestData);

        $encryptedData = $this->hesabeCrypt->encrypt(
            $requestDataJson,
            env('HESABE_SECRET_KEY'),
            env('HESABE_IV_KEY')
        );

        $client = new Client();
        try {
            // dd('Sending request to Hesabe API: ' . env('HESABE_CHECKOUT_API_URL'));
            $response = $client->request('POST', env('HESABE_CHECKOUT_API_URL'), [
                'headers' => [
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                    'accessCode' => env('HESABE_ACCESS_CODE'),
                ],
                'json' => [
                    'data' => $encryptedData,
                ],
            ]);

            $responseBody = $response->getBody()->getContents();
            $decryptedResponse = $this->hesabeCrypt->decrypt(
                $responseBody,
                env('HESABE_SECRET_KEY'),
                env('HESABE_IV_KEY')
            );
            $responseData = json_decode($decryptedResponse, true);
            // Log::info('Sending request to Hesabe API: ' . env('HESABE_CHECKOUT_API_URL'));
            // Log::info('Request data: ' . json_encode($requestData));
            if (isset($responseData['status']) && $responseData['status'] === true) {
                return response()->json([
                    'success' => true,
                    'paymentUrl' => $responseData['data'] ?? null,
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'error' => $responseData['message'] ?? 'Unknown error occurred',
                ], 400);
            }
        } catch (RequestException $e) {
            $errorMessage = $e->hasResponse() ? $e->getResponse()->getBody()->getContents() : $e->getMessage();
            return response()->json([
                'success' => false,
                'error' => 'API request failed: ' . $errorMessage,
            ], $e->getCode() ?: 500);
        }
    }

    public function paymentResponse(Request $request): JsonResponse
    {
        $decryptedData = $this->hesabeCrypt->decrypt(
            $request->input('data'),
            env('HESABE_SECRET_KEY'),
            env('HESABE_IV_KEY')
        );
        if (json_last_error() === JSON_ERROR_NONE && isset($decryptedData['status'])) {
            if ($decryptedData['status'] === 'success') {
                return response()->json([
                    'message' => 'Payment successful!',
                    'data' => $decryptedData,
                ]);
            } else {
                return response()->json([
                    'message' => 'Payment failed',
                    'data' => $decryptedData,
                ], 400);
            }
        } else {
            return response()->json([
                'message' => 'Invalid payment response',
            ], 400);
        }
    }

    public function paymentFailure(Request $request): JsonResponse
    {
        $decryptedData = $this->hesabeCrypt->decrypt(
            $request->data,
            env('HESABE_SECRET_KEY'),
            env('HESABE_IV_KEY')
        );

        return response()->json([
            'message' => 'Payment failed',
            'data' => $decryptedData,
        ], 400);
    }

    public function handleWebhook(Request $request): JsonResponse
    {
        $decryptedData = $this->hesabeCrypt->decrypt(
            $request->data,
            env('HESABE_SECRET_KEY'),
            env('HESABE_IV_KEY')
        );

        return response()->json([
            'message' => 'Webhook handled successfully!',
            'data' => $decryptedData,
        ]);
    }
}
