<?php
namespace App\Http\Controllers;

use App\Http\Requests\GetAppointmentRequest;
use App\Http\Requests\StoreAppointmentRequest;
use App\Http\Resources\AppointmentResource;
use App\Models\Appointment;
use App\Models\Doctor;
use App\Models\Service;
use App\Traits\HttpResponses;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use function PHPUnit\Framework\isEmpty;

/**
 * @group Appointment management
 *
 * APIs to manage appointment resources
 * */
class AppointmentController extends Controller
{
    use HttpResponses;

    /**
     * Show all appointments.
     *
     *
     */
    public function index()
    {
        $appointments = Appointment::with(['doctor', 'service'])->get();

        return AppointmentResource::collection($appointments);
    }
    /**
     * Store a newly created resource in storage.
     */
//    public function store(StoreAppointmentRequest $request): JsonResponse
//    {
//        // Retrieve the validated data with IDs
//        $validated = $request->validated();
//
//        // Find the doctor and service by name to get their IDs
//        $doctor = Doctor::where('name', $validated['doctor_name'])->firstOrFail();
//        $service = Service::where('name_english', $validated['service_name'])->firstOrFail();
//
//        // Create the appointment with the 'reserved' tag by default
//        $appointment = Appointment::create([
//            'doctor_id' => $doctor->id,
//            'service_id' => $service->id,
//            'date' => $validated['date'],
//            'start_time' => $validated['start_time'],
//            'end_time' => $validated['end_time'],
//            'tag' => 'reserved',
//        ]);
//
//        // Generate intervals for the given doctor and date
//        $intervals = $this->generateIntervals($doctor->id, $validated['date']);
//
//        return response()->json([
//            'status' => 'success',
//            'message' => 'Appointment created successfully.',
//            'appointment' => $appointment,
//            'intervals' => $intervals,
//        ]);
//    }
//

    /**
     * Create an appointment
     * @response 200 scenario="request was successful"
     * {
     * "status": "success",
     * "message": "Appointment created successfully.",
     * "appointment": {
     * "doctor_id": 8,
     * "service_id": 5,
     * "date": "2024-12-02",
     * "start_time": "12:30",
     * "end_time": "13:00",
     * "tag": "reserved"
     * },
     * "intervals": {
     * "headers": {},
     * "original": {
     * "intervals": [
     * {
     * "start_time": "12:15",
     * "end_time": "12:30",
     * "tag": "not reserved"
     * },
     * {
     * "start_time": "12:30",
     * "end_time": "13:00",
     * "tag": "reserved"
     * },
     * {
     * "start_time": "13:00",
     * "end_time": "13:15",
     * "tag": "not reserved"
     * }
     * ]
     * },
     * "exception": null
     * }
     * }
     *
     * @response 404 scenario="the end time of the appointment should be after the start time"{
     * "status": "error",
     * "message": "Validation failed",
     * "errors": {
     * "end_time": [
     * "The end time must be after the start time."
     * ]
     * }
     * }
     * @response 404 scenario="the date parameter should be today or a future date"{
     *  "status": "error",
     *  "message": "Validation failed",
     *  "errors": {
     *  "date": [
     *  "The appointment date must be today or a future date."
     *  ]
     *  }
     *  }
     * @bodyParam doctor_id int required The id of the doctor associated with the appointment. Example: null
     * @bodyParam service_id int required The id of the service for the appointment. Example: null
     * @bodyParam date string required The date of the appointment in 'Y-m-d' format. Example: 2024-10-01
     * @bodyParam start_time string required The start time of the appointment in 'H:i' format. Must be within the doctor's available working hours. Example: 09:00
     * @bodyParam end_time string required The end time of the appointment in 'H:i' format. Must be after the start time and within the doctor's available working hours. Example: 10:30
*/

    public function store(StoreAppointmentRequest $request): JsonResponse
    {
        // Retrieve the validated data with IDs
        $validated = $request->validated();

        $doctor = Doctor::findOrFail($validated['doctor_id'])->firstOrFail();
        $service = Service::findOrFail($validated['service_id'])->firstOrFail();

        // Check if the appointment time is within the doctor's available time
        if ($validated['start_time'] < $doctor->start_time || $validated['end_time'] > $doctor->end_time) {
            return response()->json([
                'status' => 'error',
                'message' => 'Appointment time is outside of the doctor\'s available hours.',
            ], 422);
        }

        // Check for overlapping appointments
//        $conflictingAppointment = Appointment::where('doctor_id', $doctor->id)
//            ->where('date', $validated['date'])
//            ->where(function ($query) use ($validated) {
//                $query->where(function ($q) use ($validated) {
//                    $q->where('start_time', '<', $validated['end_time'])
//                        ->where('end_time', '>', $validated['start_time']);
//                });
//            })
//            ->first();
//
//        // If there is a conflicting appointment, return an error message
//        if ($conflictingAppointment) {
//            return $this->error(null,'The appointment interferes with a previously reserved appointment.',422);
//        }

        $conflictingAppointment = Appointment::where('doctor_id', $doctor->id)
            ->where('date', $validated['date'])
            ->where(function ($query) use ($validated) {
                $query->where('start_time', '<', $validated['end_time'])
                    ->where('end_time', '>', $validated['start_time']);
            })
            ->first();

        if ($conflictingAppointment) {
            return $this->error(null, 'The appointment interferes with a previously reserved appointment.', 422);
        }


        // Create the appointment with the 'reserved' tag by default
        $appointment = Appointment::create([
            'doctor_id' => $doctor->id,
            'service_id' => $service->id,
            'date' => $validated['date'],
            'start_time' => $validated['start_time'],
            'end_time' => $validated['end_time'],
            'tag' => 'reserved',
        ]);

        // Generate intervals for the given doctor and date
        $intervals = $this->generateIntervals($doctor->id, $validated['date']);

        return response()->json([
            'status' => 'success',
            'message' => 'Appointment created successfully.',
            'appointment' => [
                'doctor_id' => $doctor->name,
                'service_id' => $service->name_english,
                'date' => $validated['date'],
                'start_time' => $validated['start_time'],
                'end_time' => $validated['end_time'],
                'tag' => 'reserved',
            ],
            'intervals' => $intervals,
        ]);
    }
//'start_time' => date('H:i', $validated['start_time']),
//'end_time' => date('H:i', $validated['end_time']),

//    public function store(StoreAppointmentRequest $request): JsonResponse
//    {
//        // Retrieve the validated data with IDs
//        $validated = $request->validated();
//
//        // Find the doctor and service by name to get their IDs
//        $doctor = Doctor::where('name', $validated['doctor_name'])->firstOrFail();
//        $service = Service::where('name_english', $validated['service_name'])->firstOrFail();
//
//        // Check if the appointment time is within the doctor's available time
//        if ($validated['start_time'] < $doctor->start_time || $validated['end_time'] > $doctor->end_time) {
//            return response()->json([
//                'status' => 'error',
//                'message' => 'Appointment time is outside of the doctor\'s available hours.',
//            ], 422);
//        }
//
//        // Create the appointment with the 'reserved' tag by default
//        $appointment = Appointment::create([
//            'doctor_id' => $doctor->id,
//            'service_id' => $service->id,
//            'date' => $validated['date'],
//            'start_time' => $validated['start_time'],
//            'end_time' => $validated['end_time'],
//            'tag' => 'reserved',
//        ]);
//
//        // Generate intervals for the given doctor and date
//        $intervals = $this->generateIntervals($doctor->id, $validated['date']);
//
//        return response()->json([
//            'status' => 'success',
//            'message' => 'Appointment created successfully.',
//            'appointment' => $appointment,
//            'intervals' => $intervals,
//        ]);
//    }



/**
 * update appointment
 * @bodyParam doctor_id int required The id of the doctor associated with the appointment. Example: null
 * @bodyParam service_id int required The id of the service for the appointment. Example: null
 * @bodyParam date string required The date of the appointment in 'Y-m-d' format. Example: 2024-10-01
 * @bodyParam start_time string required The start time of the appointment in 'H:i' format. Must be within the doctor's available working hours. Example: 09:00
 * @bodyParam end_time string required The end time of the appointment in 'H:i' format. Must be after the start time and within the doctor's available working hours. Example: 10:30
 *
 *
 * @response 404 scenario="there is no appointment with the specified id"
 * {
 * "status": "error",
 * "message": "there is no appointment with the specified id",
 * "data": ""
 * }
 * @response 404 scenario="the date parameter should be a future date"{
 * "status": "error",
 * "message": "Validation failed",
 * "errors": {
 * "date": [
 * "The appointment date must be today or a future date."
 * ]
 * }
 * }
 * @response 404 scenario="the end time of the appointment should be after the start time"{
 * "status": "error",
 * "message": "Validation failed",
 * "errors": {
 * "end_time": [
 * "The end time must be after the start time."
 * ]
 * }
 * }
 * @response 200 scenario="request has been successful"
 * {
 * "status": "Request was successful",
 * "message": "appointment has been updated successfully",
 * "data": {
 * "id": 1,
 * "doctor_id": 8,
 * "service_id": 5,
 * "date": "2024-12-06",
 * "start_time": "12:30",
 * "end_time": "13:30",
 * "tag": "reserved",
 * "created_at": "2024-10-11T22:02:41.000000Z",
 * "updated_at": "2024-10-12T00:33:44.000000Z"
 * }
 * }
 * @response 404 scenario="there is no doctor with the specified id"
 * {
 * "status": "error",
 * "message": "Validation failed",
 * "errors": {
 * "doctor_id": [
 * "The specified doctor does not exist in our records."
 * ]
 * }
 * }
 * @response 404 scenario="there is no service with the specified id"
 * {
 * "status": "error",
 * "message": "Validation failed",
 * "errors": {
 * "service_id": [
 * "The specified service does not exist in our records."
 * ]
 * }
 * }
 */
    public function update(StoreAppointmentRequest $request , $id): JsonResponse
    {
        try {

        $appointment = Appointment::findOrFail($id);
        }catch (ModelNotFoundException $e){
            return $this->error("","there is no appointment with the specified id",404);
        }
        $validated = $request->validated();

        $doctor = Doctor::findOrFail($validated['doctor_id'])->firstOrFail();
        $service = Service::findOrFail($validated['service_id'])->firstOrFail();

        // update the appointment
        $appointmentData=$appointment->update([
            'doctor_id' => $doctor->id,
            'service_id' => $service->id,
            'date' => $validated['date'],
            'start_time' => $validated['start_time'],
            'end_time' => $validated['end_time'],
            'tag' => 'reserved',
        ]);

//dd($appointment);
        return $this->success($appointment,"appointment has been updated successfully");


    }


    /**
     * Get all appointments by doctor and date
     * @queryParam doctor_id int required The id of the doctor whose appointments should be retrieved. Example: "Dr. John Doe"
     * @queryParam date date required The date for which to retrieve appointments. Example: "2024-10-15"
     * @response 200 scenario="request was successful"
     * {
     * "status": "success",
     * "message": "Appointments retrieved successfully.",
     * "appointments": [
     * {
     * "doctor_name": "mahmoud mohamed",
     * "service_name": "teeth removal",
     * "date": "2024-12-2",
     * "start_time": "12:30",
     * "end_time": "13:00"
     * }
     * ],
     * "intervals": {
     * "headers": {},
     * "original": {
     * "intervals": [
     * {
     * "start_time": "12:15",
     * "end_time": "12:30",
     * "tag": "not reserved"
     * },
     * {
     * "start_time": "12:30",
     * "end_time": "13:00",
     * "tag": "reserved"
     * },
     * {
     * "start_time": "13:00",
     * "end_time": "13:15",
     * "tag": "not reserved"
     * }
     * ]
     * },
     * "exception": null
     * }
     * }
     * @response 404 scenario="there is no doctor associated with the specifiedid"
     *  {
     *  "status": "error",
     *  "message": "There is no doctor with this id",
     *  "data": ""
     *  }
     *
     */
    public function getAppointments(Request $request): JsonResponse
    {
        // Validate only specific fields
        // $validatedData = $request->validated();

        // dd($validatedData);
        // Retrieve the validated doctor name and date
        $date = $request['date'];

        try {
            $doctor = Doctor::findOrFail($request['doctor_id']);
//            dd($doctor);
            try {
                // Find appointments for the doctor on the given date
                $appointments = Appointment::where('doctor_id', $doctor->id)
                    ->where('date', $date)
                    ->get();

                // Generate intervals for the given doctor and date

                $intervals = $this->generateIntervals($doctor->id, $date);

                $appointmentData = [];
                foreach ($appointments as $appointment) {
                    // Attempt to find the service by ID
                    try {
                        $service = Service::where('id', $appointment->service_id)->firstOrFail();
                    } catch (ModelNotFoundException $e) {
                        return $this->error("", "Service not found for one or more appointments", 404);
                    }

                    // Format the appointment data
                    $appointmentData[] = [
                        'doctor_name' => $doctor->name,

                        'service_name' => $service->name_english,
                        'date' => $date,
//                        'start_time' => $appointment['start_time'],
//                        'end_time' => $appointment['end_time'],
                        'start_time' => $appointment['start_time']->format('H:i'),
                        'end_time' => $appointment['end_time']->format('H:i'),
//                        'tag' => 'reserved',

                        // 'tag' => $appointment->tag,
                    ];
                }

                return response()->json([
                    'status' => 'success',
                    'message' => 'Appointments retrieved successfully.',
                    // 'appointments' => $appointments,
                    'appointments' => $appointmentData, // Formatted appointment data
                    'intervals' => $intervals,
                ]);
            } catch (ModelNotFoundException $e) {
                return $this->error("", "No appointments found for the given doctor on the specified date", 404);
            }

        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no doctor with this id", 404);
        }
    }



//    public function getAppointmentsByDoctor($doctor_name)
//    {
//
//        $appointment=Appointment::all();;
//        $doctor = Doctor::where('name',$doctor_name)->firstOrFail();
//        $intervals = $this->generateIntervals($doctor->id, 'date');
//        return response()->json([
//            'appointment' => $appointment,
//            'intervals' => $intervals,
//        ]);
//    }




//    public function getAppointmentsByDoctor($doctorName): JsonResponse
//    {
//        // Find the doctor by name
//        $doctor = Doctor::where('name', $doctorName)->firstOrFail();
//
//        // Retrieve all appointments for the given doctor
//        $appointments = Appointment::where('doctor_id', $doctor->id)
//            ->orderBy('date')
//            ->orderBy('start_time')
//            ->get();
//
//        // Generate intervals for the given doctor and date
//        $date = now()->format('Y-m-d'); // Or use a specific date if needed
//        $intervals = $this->generateIntervals($doctor->id, $date);
//
//        return response()->json([
//            'status' => 'success',
//            'appointments' => $appointments,
//            'intervals' => $intervals,
//        ]);
//    }

    /**
     * Get all appointments by doctor
     * @queryParam doctor_id int required The id of the doctor whose appointments should be retrieved. Example: 1
     * @response 200 scenario="request was successful"
     * {
     * "status": "Request was successful",
     * "message": "success",
     * "data": [
     * {
     * "id": 1,
     * "doctor_id": 8,
     * "service_id": 5,
     * "date": "2024-12-02",
     * "start_time": "12:30",
     * "end_time": "13:00",
     * "tag": "reserved",
     * "created_at": "2024-10-11T22:02:41.000000Z",
     * "updated_at": "2024-10-11T22:02:41.000000Z"
     * }
     * ]
     * }
     * @response 404 scenario="there is no doctor associated with the specifiedid"
     * {
     * "status": "error",
     * "message": "There is no doctor with this id",
     * "data": ""
     * }
     *
     * @response 200 scenario="request was successful but no appointments"
     * {
     * "status": "Request was successful",
     * "message": "success",
     * "data": []
     * }
    */
    public function getAppointmentsByDoctor(Request $request): JsonResponse
    {
        // Find the doctor by name
//        dd($doctorName);
//        $doctor = Doctor::findOrFail($request->doctor_id)->first();
//        if ($doctor) {
//            dd($doctor->name);
//        } else {
//            // Handle the case where no doctor was found
//            dd('Doctor not found');
//        }        // Retrieve all appointments for the given doctor

        try {
            $doctor = Doctor::findOrFail($request['doctor_id']);
//            dd($doctor);
            try {
            $appointments = Appointment::where('doctor_id',$request->doctor_id)
                ->orderBy('date')
                ->orderBy('start_time')
                ->get();
//        dd($appointments);
//        if (Appointment::where('doctor_id',$request->doctor_id)
//            ->orderBy('date')
//            ->orderBy('start_time')
//            ->first()==null) //check if there is any appointments with this doctor id
//        {
//            return $this->error("","there is no doctor with this id",404);
//        }

//        // Use the provided date or default to today
//        $date = $date ?? now()->format('Y-m-d');

        // Generate intervals for the given doctor and date
        $intervals = $this->generateIntervals($request->doctor_id,"");

        return $this->success($appointments,"success");


    } catch (ModelNotFoundException $e) {
return $this->error("", "No appointments found for the given doctor on the specified date", 404);
}

} catch (ModelNotFoundException $e) {
    return $this->error("", "There is no doctor with this id", 404);
}
    }
    /*
     * Delete appointment by id
     * @apiResourceCollection App\Http\Resources\AppointmentResource
     * @apiResourceModel App\Models\Appointment
     */
protected function destroy($id):JsonResponse{
    $appointment = Appointment::findOrFail($id);
    $appointment->delete();
    return $this->success(null,"the patient appointment has been deleted successfully");
}

    /**
     * Generate intervals for a given doctor and date.
     */
//    protected function generateIntervals($doctorId, $date): array
//    {
//        $doctor = Doctor::where('id', $doctorId)->firstOrFail();
//        // Fetch all appointments for the given doctor and date
//        $appointments = Appointment::where('doctor_id', $doctorId)
//            ->where('date', $date)
//            ->orderBy('start_time')
//            ->get();
//
//        $startOfDay = $doctor->start_time;
//        $endOfDay = $doctor->end_time;
//
//        // Initialize intervals array with the whole day as 'not reserved'
//        $intervals = [
//            [
//                'start_time' => $startOfDay,
//                'end_time' => $endOfDay,
//                'tag' => 'not reserved',
//            ]
//        ];
//
//        // Process each appointment to adjust intervals
//        foreach ($appointments as $appointment) {
//            $newIntervals = [];
//            foreach ($intervals as $interval) {
//                // If appointment starts before the interval ends and ends after the interval starts
//                if ($appointment->start_time < $interval['end_time'] && $appointment->end_time > $interval['start_time']) {
//                    // Split before the appointment if necessary
//                    if ($appointment->start_time > $interval['start_time']) {
//                        $newIntervals[] = [
//                            'start_time' => $interval['start_time'],
//                            'end_time' => $appointment->start_time,
//                            'tag' => 'not reserved',
//                        ];
//                    }
//
//                    // Mark the appointment time as 'reserved'
//                    $newIntervals[] = [
//                        'start_time' => max($interval['start_time'], $appointment->start_time),
//                        'end_time' => min($interval['end_time'], $appointment->end_time),
//                        'tag' => 'reserved',
//                    ];
//
//                    // Split after the appointment if necessary
//                    if ($appointment->end_time < $interval['end_time']) {
//                        $newIntervals[] = [
//                            'start_time' => $appointment->end_time,
//                            'end_time' => $interval['end_time'],
//                            'tag' => 'not reserved',
//                        ];
//                    }
//                } else {
//                    // If the interval is completely outside of the appointment time
//                    $newIntervals[] = $interval;
//                }
//            }
//            $intervals = $newIntervals;
//        }
//
//        // Ensure no duplicate intervals
//        $intervals = array_unique($intervals, SORT_REGULAR);
//
//        return $intervals;
//    }



    public function generateIntervals($doctorId, $date): JsonResponse
    {
        try {
            $doctor = Doctor::findOrFail($doctorId);
            $intervals = Appointment::generateIntervals($doctor->id, $date);
            return response()->json(['intervals' => $intervals], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to generate intervals: ' . $e->getMessage()], 500);
        }
    }

}
