<?php

namespace App\Http\Requests;

use Illuminate\Contracts\Validation\ValidationRule;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class StoreDoctorRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, ValidationRule|array|string>
     */
    public function rules(): array
    {
        $doctorId = $this->route('id'); // Get the doctor ID from the route if available

        return [
            'name' => [
                'required',
                'string',
                'max:255',
                $doctorId ? Rule::unique('doctors')->ignore($doctorId) : 'unique:doctors',
            ],
            'code' => [
                'required',
                'string',
                'max:255',
                $doctorId ? Rule::unique('doctors')->ignore($doctorId) : 'unique:doctors',


                /*
                 * When you are updating an existing doctor, you need to ignore the uniqueness check for the current doctor's name or code.
                 * This way, you can update the record with the same name or code without causing a validation error
                 * */
            ],


            'type' => ['required', 'string', 'max:255'],
            'specialization' => ['required', 'string', 'max:255'],
            'work_days' => ['required', 'array', 'min:1'],
            'work_days.*' => ['in:monday,tuesday,wednesday,thursday,friday,saturday,sunday'],
            'start_time' => ['required', 'date_format:H:i'],
            'end_time' => ['required', 'date_format:H:i', 'after:start_time'],
        ];
    }

    /**
     * Get the custom validation messages.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'name.unique' => 'This doctor name already exists.',
            'code.unique' => 'This doctor code already exists.',
            'name.required' => 'The name field is required.',
            'code.required' => 'The code field is required.',
            'type.required' => 'The type field is required.',
            'specialization.required' => 'The specialization field is required.',
            'work_days.required' => 'The work_days field is required.',
            'work_days.min' => 'Choose at least one working day.',
            'work_days.in' => 'A non-valid day was entered (valid days: monday, tuesday, wednesday, thursday, friday, saturday, sunday).',
            'start_time.required' => 'The start time field is required.',
            'start_time.date_format' => 'A non-valid start time format was entered (valid format e.g., 06:25, 22:50).',
            'end_time.required' => 'The end time field is required.',
            'end_time.date_format' => 'A non-valid end time format was entered (valid format e.g., 06:25, 22:50).',
            'end_time.after' => 'The end time must be after the start time.',
        ];
    }

    /**
     * Handle a failed validation attempt.
     *
     * @param Validator $validator
     * @throws HttpResponseException
     */
    public function failedValidation(Validator $validator): void
    {
        $errors = (new ValidationException($validator))->errors();
        throw new HttpResponseException(response()->json([
            'status' => 'error',
            'message' => 'Validation failed',
            'data' => $errors,
        ], 422));
    }
}
