<?php

namespace App\Http\Requests;

use Illuminate\Contracts\Validation\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Http\Exceptions\HttpResponseException;
use App\Models\Doctor;
use App\Models\Service;

class StoreAppointmentRequest extends FormRequest
{
protected $doctorId;
protected $serviceId;

/**
* Determine if the user is authorized to make this request.
*/
public function authorize(): bool
{
return true;
}

/**
* Get the validation rules that apply to the request.
*/
public function rules(): array
{
return [
'doctor_name' => ['required', 'string', 'max:255', 'exists:doctors,name'],
'service_name' => ['required', 'string', 'max:255', 'exists:services,name_english'],
'date' => ['required', 'date', 'after_or_equal:today'],
'start_time' => ['required', 'date_format:H:i'],
'end_time' => ['required', 'date_format:H:i', 'after:start_time'],
];
}

/**
* Get custom messages for validator errors.
*/
public function messages(): array
{
return [
'doctor_name.required' => "The doctor's name is required.",
'doctor_name.string' => "The doctor's name must be a string.",
'doctor_name.max' => "The doctor's name must not exceed 255 characters.",
'doctor_name.exists' => "The specified doctor does not exist in our records.",

'service_name.required' => "The service name is required.",
'service_name.string' => "The service name must be a string.",
'service_name.max' => "The service name must not exceed 255 characters.",
'service_name.exists' => "The specified service does not exist in our records.",

'date.required' => "The appointment date is required.",
'date.date' => 'The appointment date must be a valid date.',
'date.after_or_equal' => 'The appointment date must be today or a future date.',

'start_time.required' => "The start time is required.",
'start_time.date_format' => 'The start time must be in the format HH:MM.',
'end_time.required' => 'The end time is required.',
'end_time.date_format' => 'The end time must be in the format HH:MM.',
'end_time.after' => 'The end time must be after the start time.',
];
}

/**
* Handle a failed validation attempt.
*/
protected function failedValidation(Validator $validator)
{
throw new HttpResponseException(response()->json([
'status' => 'error',
'message' => 'Validation failed',
'errors' => $validator->errors()
], 422));
}

/**
* Override the validated method to include additional logic.
* @param null $key
* @param null $default
*/
public function validated($key = null, $default = null)
{
$data = parent::validated();

// Find the doctor by name and store the ID
$doctor = Doctor::where('name', $data['doctor_name'])->firstOrFail();
$this->doctorId = $doctor->id;

// Find the service by its English name and store the ID
$service = Service::where('name_english', $data['service_name'])->firstOrFail();
$this->serviceId = $service->id;

// Return validated data with additional fields
return array_merge($data, [
'doctor_id' => $this->doctorId,
'service_id' => $this->serviceId,
]);
}
}
