<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreServiceRequest;
use App\Models\Service;
use App\Traits\HttpResponses;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

/**
 * @group Services management
 *
 * APIs to manage services resources
 */
class ServiceController extends Controller
{
    use HttpResponses;

    /**
     * Get all services
     *
     * @response 200 "request was successful"{
     * "status": "Request was successful",
     * "message": "success",
     * "data": {
     * "name_english": "teeth removal",
     * "name_arabic": "خلع أسنان",
     * "code": "05304545dssd555223",
     * "QTY": "20",
     * "cost": "210",
     * "price": "260",
     * "service_group_id": "2",
     * "updated_at": "2024-10-11T07:53:18.000000Z",
     * "created_at": "2024-10-11T07:53:18.000000Z",
     * "id": 4
     * }
     * }
     */
    public function index(): JsonResponse
    {
        $services = Service::all();
        return $this->success($services, "success");
    }
//i removed integer from qty and numeric from cost in the annotation
    /**
     * Store a new service
     *
     * @bodyParam name_english string required The English name of the service. Example: "teeth removal"
     * @bodyParam name_arabic string required The Arabic name of the service. Example: "خلع أسنان"
     * @bodyParam code string required The unique service code. Example: "05304545dssd555223"
     * @bodyParam QTY optional The quantity of the service. Example: 20
     * @bodyParam cost optional The cost of the service. Example: 210.50
     * @bodyParam price numeric required The price of the service. Example: 260.00
     * @bodyParam service_group_id integer required The ID of the service group. Example: 2
     *
     * @response 200 scenario="service created successfully"{
     * "status": "Request was successful",
     * "message": "the service has been created successfully",
     * "data": {
     * "code": "4512",
     * "name": "teeth",
     * "updated_at": "2024-10-11T07:48:23.000000Z",
     * "created_at": "2024-10-11T07:48:23.000000Z",
     * "id": 5
     * }
     * }
     * @response 404 scenario="no service group with the specified id"{
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "service_group_id": [
     * "there is no service group with this id"
     * ]
     * }
     * }
     * @response 403 scenario="code or name already exists"{
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "code": [
     * "the specified code already exists"
     * ],
     * "name": [
     * "the specified name already exists"
     * ]
     * }
     * }
     */
    public function store(StoreServiceRequest $request): JsonResponse
    {
        $validatedData = $request->validated();
        $service = Service::create($validatedData);
        return $this->success($service, 'success');
    }

    /**
     * Display a service by ID
     *
     * @urlParam id integer required The ID of the service to retrieve. Example: 4
     *
     * @response 404 scenario="no service with the specified id"
     * {
     * "status": "error",
     * "message": "There is no service with this ID",
     * "data": ""
     * }
     * @response 200 scenario="request was successful"
     * {
     * "status": "Request was successful",
     * "message": "success",
     * "data": {
     * "id": 4,
     * "service_group_id": 2,
     * "code": "05304545d555223",
     * "name_english": "teeth removal",
     * "name_arabic": "خلع أسنان",
     * "QTY": 20,
     * "cost": 210,
     * "price": 260,
     * "created_at": "2024-10-11T07:53:18.000000Z",
     * "updated_at": "2024-10-11T07:53:18.000000Z"
     * }
     * }
     */
    public function show($id): JsonResponse
    {
        try {
            $service = Service::findOrFail($id);
            return $this->success($service, "success");
        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no service with this ID", 404);
        }
    }

    /**
     * Update a service
     *
     * @urlParam id integer required The ID of the service to update. Example: 5
     * @bodyParam name_english string required The English name of the service. Example: "teeth removal"
     * @bodyParam name_arabic string required The Arabic name of the service. Example: "خلع أسنان"
     * @bodyParam code string required The unique service code. Example: "05304545dssd555223"
     * @bodyParam QTY integer optional The quantity of the service. Example: 20
     * @bodyParam cost numeric optional The cost of the service. Example: 210.50
     * @bodyParam price numeric required The price of the service. Example: 260.00
     * @bodyParam service_group_id integer required The ID of the service group. Example: 2
     *
     * @response 200 scenario="service updated successfully"{
     * "status": "Request was successful",
     * "message": "Service updated successfully",
     * "data": {
     * "code": "4512",
     * "name": "teeth",
     * "updated_at": "2024-10-11T07:48:23.000000Z",
     * "created_at": "2024-10-11T07:48:23.000000Z",
     * "id": 5
     * }
     * }
     * @response 404 scenario="no service with the specified id"
     * {
     * "status": "error",
     * "message": "There is no service with this ID",
     * "data": ""
     * }
     * @response 403 scenario="code or name already exists"{
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "code": [
     * "the specified code already exists"
     * ],
     * "name": [
     * "the specified name already exists"
     * ]
     * }
     * }
     */
    public function update(StoreServiceRequest $request, $id): JsonResponse
    {
        try {
            $service = Service::findOrFail($id);
            $validatedData = $request->validated();
            $service->update($validatedData);
            return $this->success($validatedData, "Service updated successfully");
        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no service with this ID", 404);
        }
    }

    /**
     * Delete a service
     *
     * @urlParam id integer required The ID of the service to delete. Example: 4
     *
     * @response 200 scenario="service deleted successfully"{
     * "status": "Request was successful",
     * "message": "the service has been deleted successfully",
     * "data": null
     * }
     * @response 404 scenario="no service with the specified id"
     * {
     * "status": "error",
     * "message": "There is no service with this ID",
     * "data": ""
     * }
     */
    public function destroy($id): JsonResponse
    {
        try {
            $service = Service::findOrFail($id);
            $service->delete();
            return $this->success(null, "the service has been deleted successfully");
        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no service with this ID", 404);
        }
    }
}
