<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreClinicRequest;
use App\Http\Resources\ClinicCollection;
use App\Http\Resources\ClinicResource;
use App\Models\Clinic;
use App\Traits\HttpResponses;
use http\Env\Response;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;


/**
 * @group clinic management
 *
 * APIs to manage clinic resources
 * */
class ClinicController extends Controller
{
    use HttpResponses;
    /**
     * Display all available clinics
     * @return JsonResponse
     * @response 200 scenario="request was successful"
     * {
     * "status": "Request was successful",
     * "message": "success",
     * "data": [
     * {
     * "id": 5,
     * "name_english": "surgery",
     * "name_arabic": "جراحه",
     * "specialization": "surgery",
     * "location": "kweit",
     * "address": "15 wall street",
     * "phone": "0904562185",
     * "mobile": "01236954812",
     * "logo": "surgery/surgery_logo.jpg",
     * "created_at": "2024-10-12T10:55:24.000000Z",
     * "updated_at": "2024-10-12T10:55:24.000000Z"
     * }
     * ]
     * }
     */
    public function index(): JsonResponse
    {
        $clinic= Clinic::all();
        return $this->success(new ClinicCollection($clinic),"success");
    }

    /**
     *Create new clinic
     *
     * @bodyParam name_english string required The English name of the clinic. Example: surgery
     * @bodyParam name_arabic string required The Arabic name of the clinic. Example: جراحه
     * @bodyParam specialization string required The specialization of the clinic. Example: surgery
     * @bodyParam location string required The location of the clinic. Example: kuwait
     * @bodyParam address string required The address of the clinic. Example: 15 wall street
     * @bodyParam phone string required The phone number of the clinic. Example: 0904562185
     * @bodyParam mobile string required The mobile number of the clinic. Example: 01236954812
     * @response 200 scenario="clinic created successfully"
     * {
     * "status": "Request was successful",
     * "message": "Clinic has been created successfully",
     * "data": {
     * "id": 3,
     * "name_english": "surgeryy",
     * "name_arabic": "جراحهi",
     * "specialization": "surgery",
     * "location": "kweit",
     * "address": "15 wall street",
     * "phone": "0904562185",
     * "mobile": "01236954812",
     * "logo": null,
     * "created_at": "2024-10-19T12:16:46.000000Z",
     * "updated_at": "2024-10-19T12:16:46.000000Z"
     * }
     * }
     * @response 422 scenario="missing attributes"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "name_english": [
     * "The english name field is required."
     * ],
     * "name_arabic": [
     * "The arabic name field is required."
     * ],
     * "specialization": [
     * "The specialization field is required."
     * ],
     * "location": [
     * "The location field is required."
     * ],
     * "address": [
     * "The address field is required."
     * ],
     * "phone": [
     * "The phone field is required."
     * ]
     * }
     * }
     *
     * @response 404 scenario="name_english or name_arabic already exists"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "name_english": [
     * "This clinic english name already exists."
     * ],
     * "name_arabic": [
     * "This clinic arabic name already exists."
     * ]
     * }
     * }
     */

    public function store(StoreClinicRequest $request): JsonResponse
    {
//        dd($request);
        $validatedData = $request->validated();
        // Handle logo upload
        $logoPath = $this->handleLogoUpload($request, $validatedData['name_english']);
//        dd($logoPath);
        // Create the clinic record
        $clinic = Clinic::create([
            'name_english' => $validatedData['name_english'],
            'name_arabic' => $validatedData['name_arabic'],
            'specialization' => $validatedData['specialization'],
            'location' => $validatedData['location'],
            'address' => $validatedData['address'],
            'phone' => $validatedData['phone'],
            'mobile' => $validatedData['mobile'],
            'logo' => $logoPath, // Save the logo path if provided
        ]);
//        dd($clinic);
        return $this->success(new ClinicResource($clinic), "Clinic has been created successfully");
    }

    /**
     * Update clinic.
     *
     * @param int $id The ID of the clinic to update.
     * @bodyParam name_english string required The English name of the clinic. Example: brain surgery
     * @bodyParam name_arabic string required The Arabic name of the clinic. Example: جراحة المخ و الأعصاب
     * @bodyParam specialization string required The specialization of the clinic. Example: surgery
     * @bodyParam location string required The location of the clinic. Example: egypt
     * @bodyParam address string required The address of the clinic. Example: 15 elnaser street
     * @bodyParam phone string required The phone number of the clinic. Example: 0123598888
     * @bodyParam mobile string required The mobile number of the clinic. Example: 01236954812
     * @bodyParam logo file The logo of the clinic.
     *
     *
     * @response 200 scenario="clinic updated successfully"
     * {
     * "status": "Request was successful",
     * "message": "Clinic updated successfully",
     * "data": {
     * "id": 1,
     * "name_english": "brain surgery",
     * "name_arabic": "جراحة المخ و الأعصاب",
     * "specialization": "surgery",
     * "location": "egypt",
     * "address": "15 elnaser street",
     * "phone": "0123598888",
     * "mobile": "01236954812",
     * "logo": "brain surgery/brain_surgery_logo.jpg",
     * "created_at": "2024-10-12T10:35:36.000000Z",
     * "updated_at": "2024-10-12T10:41:30.000000Z"
     * }
     * }
     * @response 404 scenario="no clinic with the specified id"
     * {
     * "status": "error",
     * "message": "There is no clinic with this ID",
     * "data": ""
     * }
     * @response 404 scenario="clinic name_english or name_arabic already exists"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "name_english": [
     * "This clinic english name already exists."
     * ],
     * "name_arabic": [
     * "This clinic arabic name already exists."
     * ]
     * }
     * }
     *
     * @response 422 scenario="missing required attributes"
     *{
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "name_english": [
     * "The english name field is required."
     * ],
     * "name_arabic": [
     * "The arabic name field is required."
     * ],
     * "specialization": [
     * "The specialization field is required."
     * ],
     * "location": [
     * "The location field is required."
     * ],
     * "address": [
     * "The address field is required."
     * ],
     * "phone": [
     * "The phone field is required."
     * ]
     * }
     */
    public function update(StoreClinicRequest $request, $id): JsonResponse
    {
        try {

            $clinic = Clinic::findOrFail($id);
            $validatedData = $request->validated();

            // Handle logo upload
            $logoPath = $this->handleLogoUpload($request, $validatedData['name_english'], $clinic->logo);

            // Update the clinic record
            $clinic->update([
                'name_english' => $validatedData['name_english'],
                'name_arabic' => $validatedData['name_arabic'],
                'specialization' => $validatedData['specialization'],
                'location' => $validatedData['location'],
                'address' => $validatedData['address'],
                'phone' => $validatedData['phone'],
                'logo' => $logoPath,
            ]);

            return $this->success(new ClinicResource($clinic), "Clinic updated successfully");

        } catch (ModelNotFoundException $e) {
            // If no service is found, return a 404 error response
            return $this->error("", "There is no clinic with this ID", 404);
        }



    }

    /**
     * show a clinic by id
     * @param $id
     * @return JsonResponse
     * @response 404 scenario="no clinic with the specified id"
     * {
     * "status": "error",
     * "message": "There is no clinic with this ID",
     * "data": ""
     * }
     * @response 200 scenario="request was successful"
     * {
     * "id": 5,
     * "name_english": "surgery",
     * "name_arabic": "جراحه",
     * "specialization": "surgery",
     * "location": "kweit",
     * "address": "15 wall street",
     * "phone": "0904562185",
     * "mobile": "01236954812",
     * "logo": "surgery/surgery_logo.jpg",
     * "created_at": "2024-10-12T10:55:24.000000Z",
     * "updated_at": "2024-10-12T10:55:24.000000Z"
     * }
     */
    public function show($id): JsonResponse
    {
        try {
        $clinic=Clinic::findOrFail($id);
        return $this->success(new ClinicResource($clinic),"clinic returned successfully");
        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no clinic with this ID", 404);
        }

    }
    /**
     * Delete Clinic
     * @response 200 scenario="clinic has been deleted successfully"
     * {
     * "status": "Request was successful",
     * "message": "the clinic has been deleted successfully",
     * "data": ""
     * }
     * @response 404 scenario="no clinic with the specified id"
     * {
     * "status": "error",
     * "message": "There is no service with this ID",
     * "data": ""
     * }
    */
    public function destroy($id):JsonResponse
    {


        try {

        $clinic = Clinic::findOrFail($id);
        $clinic->delete();
        return $this->success('',"the clinic has been deleted successfully");

        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no service with this ID", 404);
        }
    }


    protected function handleLogoUpload(Request $request, string $nameEnglish, string $currentLogoPath = null): ?string
    {

        //$logoName = Str::replace(' ', '_', trim(pathinfo($validatedData['logo']->getClientOriginalName(), PATHINFO_FILENAME)));
        /*
            pathinfo($validatedData['logo'] //this gets the name that the file was uploaded by
            The pathinfo() function returns information about a file path.
            By passing PATHINFO_FILENAME as the second argument, it specifically extracts just the filename without the extension.
            Example: If the filename is "my-logo.png", pathinfo($validatedData['logo']->getClientOriginalName(), PATHINFO_FILENAME) will return "my-logo".
        */

        if ($request->hasFile('logo')) {
            // Delete old logo if it exists
            if ($currentLogoPath && Storage::disk('logos')->exists($currentLogoPath)) {
                Storage::disk('logos')->delete($currentLogoPath);
            }

            // Store new logo
            $logoName = Str::replace(' ', '_', trim($nameEnglish));
            $fileName = $logoName . "_logo." . $request->file('logo')->getClientOriginalExtension();
            return $request->file('logo')->storeAs($nameEnglish, $fileName, 'logos');
        }

        return $currentLogoPath; // Return current path if no new logo is provided
    }




//    public function store(StoreClinicRequest $request): JsonResponse
//    {
//        $validatedData = $request->validated();
//
////            dd($validatedData['name_english']);
//        // Handle logo upload
//        if ($request->hasFile('logo')) {
//            //$logoName = Str::replace(' ', '_', trim(pathinfo($validatedData['logo']->getClientOriginalName(), PATHINFO_FILENAME)));
//            /*
//                pathinfo($validatedData['logo'] //this gets the name that the file was uploaded by
//                The pathinfo() function returns information about a file path.
//                By passing PATHINFO_FILENAME as the second argument, it specifically extracts just the filename without the extension.
//                Example: If the filename is "my-logo.png", pathinfo($validatedData['logo']->getClientOriginalName(), PATHINFO_FILENAME) will return "my-logo".
//            */
//            $logoName = Str::replace(' ', '_', trim($validatedData['name_english']));
//            $fileName = $logoName. "_logo." . $validatedData['logo']->getClientOriginalExtension();
//            $logoPath = $validatedData['logo']->storeAs($validatedData['name_english'], $fileName, 'logos');
////            dd($logoName);
////            dd($fileName);
////            dd($logoPath);
//        } else {
//            $logoPath = null; // Handle the case where no logo is provided
//        }
//
//        // Create the clinic record
//        $clinic = Clinic::create([
//            'name_english' => $validatedData['name_english'],
//            'name_arabic' => $validatedData['name_arabic'],
//            'specialization' => $validatedData['specialization'],
//            'location' => $validatedData['location'],
//            'address' => $validatedData['address'],
//            'phone' => $validatedData['phone'],
//            'mobile' => $validatedData['mobile'],
//            'logo' => $logoPath, // Save the icon path if provided
//        ]);
//
//        return $this->success($clinic, "Clinic created successfully");
//    }


    /**
     * Update the specified resource in storage.
     */
//    public function update(StoreClinicRequest $request, $id)
//    {
//        $clinic = Clinic::findOrFail($id);
//        $validatedData = $request->validated();
//
//        if ($request->hasFile('logo')) {
//            //$logoName = Str::replace(' ', '_', trim(pathinfo($validatedData['logo']->getClientOriginalName(), PATHINFO_FILENAME)));
//            /*
//                pathinfo($validatedData['logo'] //this gets the name that the file was uploaded by
//                The pathinfo() function returns information about a file path.
//                By passing PATHINFO_FILENAME as the second argument, it specifically extracts just the filename without the extension.
//                Example: If the filename is "my-logo.png", pathinfo($validatedData['logo']->getClientOriginalName(), PATHINFO_FILENAME) will return "my-logo".
//            */
//            $logoName = Str::replace(' ', '_', trim($validatedData['name_english']));
//            $fileName = $logoName. "_logo." . $validatedData['logo']->getClientOriginalExtension();
//            $logoPath = $validatedData['logo']->storeAs($validatedData['name_english'], $fileName, 'logos');
////            dd($logoName);
////            dd($fileName);
////            dd($logoPath);
//        } else {
//            $logoPath = null; // Handle the case where no logo is provided
//        }
//
//
//
//
//        $clinic->update([
//            'english_name'=>$validatedData['name_english'],
//            'name_arabic'=>$validatedData['name_arabic'],
//            'specialization'=>$validatedData['specialization'],
//            'location'=>$validatedData['location'],
//            'address'=>$validatedData['address'],
//            'phone'=>$validatedData['phone'],
//            'logo'=>$logoPath,
//        ]);
//    }

    /**
     * Remove the specified resource from storage.
     */

}
