<?php

namespace App\Http\Controllers;

use App\Http\Requests\LoginUserRequest;
use App\Http\Requests\ResetPasswordRequest;
use App\Http\Requests\StoreUserRequest;
use App\Models\User;
use App\Traits\HttpResponses;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

/**
 * @group Auth management
 *
 * APIs to manage authentication
 */
class AuthController extends Controller
{
    use HttpResponses;

    /**
     * Login a user
     *
     * @param LoginUserRequest $request
     *
     * @bodyParam name string required The username of the user. Example: ahmed
     * @bodyParam password string required The user's password. Example: password123
     *
     * @return \Illuminate\Http\JsonResponse
     *
     * @response 200 scenario="user logged in successfully"
     * {
     * "status": "Request was successful",
     * "message": "user logged in successfully",
     * "data": {
     * "user": {
     * "id": 1,
     * "name": "ahmed ismail",
     * "created_at": "2024-10-12T22:04:34.000000Z",
     * "updated_at": "2024-10-12T22:04:34.000000Z"
     * },
     * "token": "5|xK7B7IWh0cEdrEwTJCQJwGWUnsEvMTc6QxkAFFVz59a8a498"
     * }
     * }
     * @response 401 scenario="wrong password"
     * {
     * "status": "error",
     * "message": "The password isn't correct",
     * "data": null
     * }
     * @response 404 scenario="user doesn't exist"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "name": [
     * "This user doesn't exist."
     * ]
     * }
     * }
     */
    public function login(LoginUserRequest $request): JsonResponse
    {
        $credentials = $request->only('name', 'password');

        if (!Auth::attempt($credentials)) {
            return $this->error(null, "The password isn't correct", 401);
        }

        $user = Auth::user();
        return $this->success([
            'user' => $user,
            'token' => $user->createToken('Token of ' . $user->name)->plainTextToken,
        ], "User logged in successfully");
    }

    /**
     * Register a new user
     *
     *
     * @bodyParam name string required The username of the user. Example: ahmed
     * @bodyParam password string required The password of the user. Example: none
     * @bodyParam password_confirmation string required The password confirmation. Example: none
     *
     * @return JsonResponse
     *
     * @response 200 scenario="user has been created successfully"
     * {
     * "status": "Request was successful",
     * "message": "user has been created successfully",
     * "data": {
     * "user": {
     * "name": "ahmed",
     * "updated_at": "2024-10-12T22:07:57.000000Z",
     * "created_at": "2024-10-12T22:07:57.000000Z",
     * "id": 3
     * },
     * "token": "3|vUhBeVm8bRTlKl8UsgInOL7p6RGvHiqeuxZYpngJ5e93c06e"
     * }
     * }
     * @response 404 scenario="username has already been taken"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "name": [
     * "This username has already been taken."
     * ]
     * }
     * }
     */
    public function register(StoreUserRequest $request): JsonResponse
    {
        $request->validate([
            'password' => 'required|confirmed', // Validate password matches password_confirmation
        ]);

        $user = User::create([
            'name' => $request->name,
            'password' => Hash::make($request->password),
        ]);

        return $this->success([
            'user' => $user,
            'token' => $user->createToken('Token of ' . $user->name)->plainTextToken,
        ], "User has been created successfully");
    }

    /**
     * Reset user password
     *
     * @param ResetPasswordRequest $request
     *
     * @bodyParam name string required The username of the user. Example: ahmed
     * @bodyParam old_password string required The old password of the user. Example: oldPassword123
     * @bodyParam new_password string required The new password of the user. Example: newPassword456
     * @bodyParam new_password_confirmation string required The password confirmation for the new password. Example: newPassword456
     *
     * @return JsonResponse
     *
     * @response 200 scenario="password reset successfully"
     * {
     * "status": "Request was successful",
     * "message": "Password has been successfully updated.",
     * "data": null
     * }
     * @response 401 scenario="incorrect old password"
     * {
     * "status": "error",
     * "message": "The old password isn't correct.",
     * "data": null
     * }
     * @response 404 scenario="new password confirmation does not match"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "new_password": [
     * "The new password confirmation does not match."
     * ]
     * }
     * }
     */
    public function resetPassword(ResetPasswordRequest $request): JsonResponse
    {
        $user = User::where('name', $request->name)->first();

        if (!Hash::check($request->old_password, $user->password)) {
            return $this->error(null, "The old password isn't correct.", 401);
        }

        $user->password = Hash::make($request->new_password);
        $user->save();

        return $this->success(null, 'Password has been successfully updated.');
    }

    /**
     * Logout the current user
     *
     * @return JsonResponse
     *
     * @response 200 scenario="logged out successfully"
     * {
     * "status": "Request was successful",
     * "message": "Logged out successfully",
     * "data": null
     * }
     * @response 404 scenario="unauthenticated"
     * {
     * "message": "Unauthenticated."
     * }
     */
    public function logout(): JsonResponse
    {
        // Revoke the current user's token
        try {
            Auth::user()->currentAccessToken()->delete();
        } catch (\Exception $e) {
            return $this->error(null, "This user either has no account or didn't log in", 404);
        }

        // Return a success response
        return $this->success(null, "Logged out successfully");
    }
}
